@extends('layouts.app')

@section('center')

<link rel="stylesheet" href="{{ asset('assets/vendor/bs-select/bs-select.css') }}" />
<link rel="stylesheet" href="{{ asset('assets/vendor/datepicker/css/classic.css') }}" />
<link rel="stylesheet" href="{{ asset('assets/vendor/datepicker/css/classic.date.css') }}" />

<div class="row gutters">
    <div class="col-xl-12 col-lg-12 col-md-12 col-sm-12 col-12">
        <form action="{{ route('timeDomainGraph') }}">
            <div class="row gutters">
                <div class="col-xl-2 col-lg-2 col-md-2 col-sm-2">
                    <div class="form-group">
                        <label class="label">Device ID</label>
                        <select class="form-control selectpicker" data-live-search="true" name="DeviceId">
                            <option value=''>Select Device ID</option>
                            @foreach($tr_data as $row)
                            <option value="{{ $row->global_device_id }}"
                                {{ request('DeviceId') == $row->global_device_id ? 'selected' : '' }}>
                                {{ $row->global_device_id }}
                            </option>
                            @endforeach
                        </select>
                    </div>
                </div>

                <div class="col-xl-2 col-lg col-md-2 col-sm-2 col-12 ">
                    <div class="form-group">
                        <label class="label">&nbsp;</label>
                        <button type="submit" class="form-control btn btn-primary generate">Generate</button>
                    </div>
                </div>
            </div>
        </form>
    </div>

    @if(!empty($graph_data))
        @foreach($graph_data as $channelGroup => $groupSamples)
            <h4>Channel Group {{ $channelGroup }}</h4><br/>
            <div class="row">
                <!-- Time-domain voltage chart -->
                <div class="col-md-6 mb-4">
                    <canvas id="vChart-{{ $channelGroup }}" width="600" height="400"></canvas>
                </div>
                <!-- Time-domain current chart -->
                <div class="col-md-6 mb-4">
                    <canvas id="iChart-{{ $channelGroup }}" width="600" height="400"></canvas>
                </div>
                <!-- Frequency-domain voltage chart -->
                <div class="col-md-6 mb-4">
                    <canvas id="vFreqChart-{{ $channelGroup }}" width="600" height="400"></canvas>
                </div>
                <!-- Frequency-domain current chart -->
                <div class="col-md-6 mb-4">
                    <canvas id="iFreqChart-{{ $channelGroup }}" width="600" height="400"></canvas>
                </div>
            </div>
        @endforeach
    @endif
</div>

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chartjs-adapter-date-fns@latest"></script>
<script src="{{ asset('assets/vendor/bs-select/bs-select.min.js') }}"></script>

<script>
const FUNDAMENTAL_FREQUENCY = 50; // Hz
const MAX_FFT_DISPLAY_FREQUENCY = 1000; // Hz limit

// Consistent color scheme
const COLOR_MAP = {
    1: 'red',     // Phase 1
    2: 'yellow',  // Phase 2
    3: 'blue'     // Phase 3
};

function extractFullCycle(inputData) {
    const ZERO_THRESHOLD = 20;
    let startIndex = -1, endIndex = -1, state = 'SEARCHING_FOR_START';
    for (let i = 1; i < inputData.length; i++) {
        const prev = inputData[i - 1], curr = inputData[i];
        if (state === 'SEARCHING_FOR_START' && prev < ZERO_THRESHOLD && curr >= ZERO_THRESHOLD) {
            startIndex = i; state = 'SEARCHING_FOR_END';
        } else if (state === 'SEARCHING_FOR_END' && prev >= ZERO_THRESHOLD && curr < ZERO_THRESHOLD) {
            endIndex = i; break;
        }
    }
    if (startIndex === -1 || endIndex === -1 || startIndex >= endIndex) return null;
    const positiveLobe = inputData.slice(startIndex, endIndex);
    const negativeLobe = positiveLobe.map(v => -v);
    const fullCycleData = positiveLobe.concat(negativeLobe);
    const fullCycleLabels = Array.from({ length: fullCycleData.length }, (_, i) => i + 1);
    return { cycleData: fullCycleData, labels: fullCycleLabels };
}

function applyFFT(timeDomainData, fundamentalFrequency) {
    const n = timeDomainData.length;
    let frequencies = [], amplitudes = [];
    if (n === 0) return { frequencies, amplitudes };
    for (let k = 0; k < n; k++) {
        let re = 0, im = 0;
        for (let t = 0; t < n; t++) {
            const angle = (2 * Math.PI * k * t) / n;
            re += timeDomainData[t] * Math.cos(angle);
            im -= timeDomainData[t] * Math.sin(angle);
        }
        frequencies[k] = k * fundamentalFrequency;
        amplitudes[k] = Math.sqrt(re * re + im * im) / n;
    }
    return { frequencies, amplitudes };
}

function safeSlice(arr, endIndex) {
    if (!arr || arr.length === 0) return [];
    return arr.slice(0, Math.min(endIndex, arr.length));
}

const allSamples = @json($graph_data ?? []);

Object.entries(allSamples).forEach(([channelGroup, groupSamples]) => {
    const v1_raw = groupSamples.map(s => s.v1);
    const v2_raw = groupSamples.map(s => s.v2);
    const v3_raw = groupSamples.map(s => s.v3);
    const i1_raw = groupSamples.map(s => s.i1);
    const i2_raw = groupSamples.map(s => s.i2);
    const i3_raw = groupSamples.map(s => s.i3);

    const v1_processed = extractFullCycle(v1_raw);
    const v2_processed = extractFullCycle(v2_raw);
    const v3_processed = extractFullCycle(v3_raw);
    const i1_processed = extractFullCycle(i1_raw);
    const i2_processed = extractFullCycle(i2_raw);
    const i3_processed = extractFullCycle(i3_raw);

    // TIME-DOMAIN VOLTAGE
    const vCtx = document.getElementById(`vChart-${channelGroup}`);
    if (vCtx && v1_processed) {
        new Chart(vCtx, {
            type: 'line',
            data: {
                labels: v1_processed.labels,
                datasets: [
                    { label: 'V1', data: v1_processed.cycleData, borderColor: COLOR_MAP[1], fill: false, pointRadius: 0 },
                    { label: 'V2', data: v2_processed ? v2_processed.cycleData : [], borderColor: COLOR_MAP[2], fill: false, pointRadius: 0 },
                    { label: 'V3', data: v3_processed ? v3_processed.cycleData : [], borderColor: COLOR_MAP[3], fill: false, pointRadius: 0 }
                ]
            },
            options: {
                responsive: true,
                plugins: { title: { display: true, text: `Synthesized Voltage Cycle (Group ${channelGroup})` }},
                scales: {
                    x: { title: { display: true, text: 'Sample Number' }},
                    y: { title: { display: true, text: 'ADC Value' }}
                }
            }
        });
    }

    // TIME-DOMAIN CURRENT
    const iCtx = document.getElementById(`iChart-${channelGroup}`);
    if (iCtx && i1_processed) {
        new Chart(iCtx, {
            type: 'line',
            data: {
                labels: i1_processed.labels,
                datasets: [
                    { label: 'I1', data: i1_processed.cycleData, borderColor: COLOR_MAP[1], fill: false, pointRadius: 0 },
                    { label: 'I2', data: i2_processed ? i2_processed.cycleData : [], borderColor: COLOR_MAP[2], fill: false, pointRadius: 0 },
                    { label: 'I3', data: i3_processed ? i3_processed.cycleData : [], borderColor: COLOR_MAP[3], fill: false, pointRadius: 0 }
                ]
            },
            options: {
                responsive: true,
                plugins: { title: { display: true, text: `Synthesized Current Cycle (Group ${channelGroup})` }},
                scales: {
                    x: { title: { display: true, text: 'Sample Number' }},
                    y: { title: { display: true, text: 'ADC Value' }}
                }
            }
        });
    }

    // Apply FFT
    const v1_fft = applyFFT(v1_processed ? v1_processed.cycleData : [], FUNDAMENTAL_FREQUENCY);
    const v2_fft = applyFFT(v2_processed ? v2_processed.cycleData : [], FUNDAMENTAL_FREQUENCY);
    const v3_fft = applyFFT(v3_processed ? v3_processed.cycleData : [], FUNDAMENTAL_FREQUENCY);
    const i1_fft = applyFFT(i1_processed ? i1_processed.cycleData : [], FUNDAMENTAL_FREQUENCY);
    const i2_fft = applyFFT(i2_processed ? i2_processed.cycleData : [], FUNDAMENTAL_FREQUENCY);
    const i3_fft = applyFFT(i3_processed ? i3_processed.cycleData : [], FUNDAMENTAL_FREQUENCY);

    // FREQUENCY-DOMAIN VOLTAGE
    const vFreqCtx = document.getElementById(`vFreqChart-${channelGroup}`);
    if (vFreqCtx && v1_fft.amplitudes.length > 0) {
        const N_v1 = v1_fft.amplitudes.length;
        let displayPointsCount = Math.floor(MAX_FFT_DISPLAY_FREQUENCY / FUNDAMENTAL_FREQUENCY) + 1;
        displayPointsCount = Math.min(displayPointsCount, Math.floor(N_v1 / 2) + 1);
        const chartLabels = v1_fft.frequencies.slice(0, displayPointsCount);
        if (chartLabels.length > 0) {
            new Chart(vFreqCtx, {
                type: 'bar',
                data: {
                    labels: chartLabels,
                    datasets: [
                        { label: 'V1 FFT', data: safeSlice(v1_fft.amplitudes, displayPointsCount), backgroundColor: COLOR_MAP[1] },
                        { label: 'V2 FFT', data: safeSlice(v2_fft.amplitudes, displayPointsCount), backgroundColor: COLOR_MAP[2] },
                        { label: 'V3 FFT', data: safeSlice(v3_fft.amplitudes, displayPointsCount), backgroundColor: COLOR_MAP[3] }
                    ]
                },
                options: {
                    responsive: true,
                    plugins: { title: { display: true, text: `Voltage FFT (Group ${channelGroup})` }},
                    scales: {
                        x: { title: { display: true, text: 'Frequency (Hz)' }},
                        y: { title: { display: true, text: 'Amplitude' }}
                    }
                }
            });
        }
    }

    // FREQUENCY-DOMAIN CURRENT
    const iFreqCtx = document.getElementById(`iFreqChart-${channelGroup}`);
    if (iFreqCtx && i1_fft.amplitudes.length > 0) {
        const N_i1 = i1_fft.amplitudes.length;
        let displayPointsCount = Math.floor(MAX_FFT_DISPLAY_FREQUENCY / FUNDAMENTAL_FREQUENCY) + 1;
        displayPointsCount = Math.min(displayPointsCount, Math.floor(N_i1 / 2) + 1);
        const chartLabels = i1_fft.frequencies.slice(0, displayPointsCount);
        if (chartLabels.length > 0) {
            new Chart(iFreqCtx, {
                type: 'bar',
                data: {
                    labels: chartLabels,
                    datasets: [
                        { label: 'I1 FFT', data: safeSlice(i1_fft.amplitudes, displayPointsCount), backgroundColor: COLOR_MAP[1] },
                        { label: 'I2 FFT', data: safeSlice(i2_fft.amplitudes, displayPointsCount), backgroundColor: COLOR_MAP[2] },
                        { label: 'I3 FFT', data: safeSlice(i3_fft.amplitudes, displayPointsCount), backgroundColor: COLOR_MAP[3] }
                    ]
                },
                options: {
                    responsive: true,
                    plugins: { title: { display: true, text: `Current FFT (Group ${channelGroup})` }},
                    scales: {
                        x: { title: { display: true, text: 'Frequency (Hz)' }},
                        y: { title: { display: true, text: 'Amplitude' }}
                    }
                }
            });
        }
    }
});
</script>
@endpush
