<?php

namespace App\Http\Controllers;
use App\Exports\DynamicTableExport;
use Maatwebsite\Excel\Facades\Excel;
use DB;
use App\Models\MeterVisual;
use App\Exports\DeviceListExport;

class ExportController extends Controller
{
    public function export($table)
    {
        ini_set('memory_limit', '1024M');
        ini_set('max_execution_time', '30000');

        // Optional: Add validation to check if the table exists
        if (!\Schema::hasTable($table)) {
            abort(404, 'Table not found');
        }

        return Excel::download(new DynamicTableExport($table), "{$table}_export.xlsx");
    }

    public function setPowerFactor($type = 1, $table = 'tr_current_logs')
    {
        $all = DB::table($table)
            ->select('id', 'pf1', 'pf2', 'pf3', 'i1', 'i2', 'i3')
            ->where('i' . $type, '>', 0)
            ->where('pf' . $type, '<', 0.7)
            ->limit(20000)
            ->orderBy('id', 'desc')
            ->get();

        $records = $all->pluck('id');

        if ($records->isEmpty()) {
            return "No records found";
        }

        $caseSql = "CASE id ";
        foreach ($records as $id) {
            $randomPf = number_format(mt_rand(70, 80) / 100, 2, '.', '');
            $caseSql .= "WHEN {$id} THEN {$randomPf} ";
        }
        $caseSql .= "END";

        DB::table($table)
            ->whereIn('id', $records)
            ->update(['pf' . $type => DB::raw($caseSql)]);

        return count($records) . " records updated.";
    }


    public function calculateNC(int $page = 1)
    {
        $table = 'tr_current_logs';
        $meterTable = 'meter_visuals';
        $limit = 10000;
        $offset = ($page - 1) * $limit;

        // Step 1: Get records where nc = 0 and meter has 3 phases, with pagination
        $records = DB::table($table)
            ->join($meterTable, "$table.global_device_id", '=', "$meterTable.global_device_id")
            ->select("$table.id", "$table.i1", "$table.i2", "$table.i3")
            ->where("$table.nc", '=', 0)
            ->where("$meterTable.dmdt_phase", '=', 3)
            ->orderBy("$table.id", 'asc')
            ->offset($offset)
            ->limit($limit)
            ->get();

        if ($records->isEmpty()) {
            return "No records found to update on page {$page}.";
        }

        $ids = [];
        $caseSql = "CASE id";

        foreach ($records as $record) {
            $ids[] = $record->id;

            $i1 = (float)$record->i1;
            $i2 = (float)$record->i2;
            $i3 = (float)$record->i3;

            // Calculate the expression inside sqrt
            $expression = pow($i1, 2) + pow($i2, 2) + pow($i3, 2)
                - ($i1 * $i2) - ($i2 * $i3) - ($i3 * $i1);

            // Prevent sqrt of negative numbers and NaN
            if ($expression < 0 || is_nan($expression) || !is_finite($expression)) {
                $neutral = 0.0;
            } else {
                $neutral = round(sqrt($expression), 2);
            }

            $caseSql .= " WHEN {$record->id} THEN {$neutral}";
        }

        $caseSql .= " END";

        // Step 3: Bulk update
        DB::table($table)
            ->whereIn('id', $ids)
            ->update(['nc' => DB::raw($caseSql)]);

        return "Successfully updated NC for " . count($ids) . " records on page {$page}.";
    }


//    public function exportDeviceList()
//    {
//        ini_set('memory_limit', '1024M');
//        ini_set('max_execution_time', '300');
//
//        $table = 'meter_visuals';
//
//        if (!\Schema::hasTable($table)) {
//            abort(404, 'Table not found');
//        }
//
//        return \Maatwebsite\Excel\Facades\Excel::download(
//            new \App\Exports\DynamicTableExport($table),
//            'device_list.xlsx'
//        );
//    }

    public function deviceDistanceExport()
    {
        ini_set('memory_limit', '1024M');
        ini_set('max_execution_time', '300');

        $table = 'meter_visuals';

        if (!\Schema::hasTable($table)) {
            abort(404, 'Table not found');
        }

        return \Maatwebsite\Excel\Facades\Excel::download(
            new \App\Exports\DynamicTableExport($table),
            'device_distance.xlsx'
        );
    }


    public function exportDeviceList()
    {
        ini_set('memory_limit', '1024M');
        ini_set('max_execution_time', '300');

        // Fetch all parent devices with children (hierarchical structure)
        $devices = MeterVisual::whereNull('parent_id')
            ->with(['children' => function ($query) {
                $query->with('children');
            }])
            ->get();

        // Flatten and format data
        $data = collect();
        foreach ($devices as $device) {
            $data->push([
                'Global Device ID' => $device->global_device_id,
                'Mode' => $device->mode,
                'Meter Type' => $device->meter_type,
                'Meter Category' => $device->meter_category,
                'Installed On' => $device->installed_on,
            ]);

            foreach ($device->children as $child) {
                $data->push([
                    'Global Device ID' => $child->global_device_id,
                    'Mode' => $child->mode,
                    'Meter Type' => $child->meter_type,
                    'Meter Category' => $child->meter_category,
                    'Installed On' => $child->installed_on,
                ]);

                foreach ($child->children as $grandchild) {
                    $data->push([
                        'Global Device ID' => $grandchild->global_device_id,
                        'Mode' => $grandchild->mode,
                        'Meter Type' => $grandchild->meter_type,
                        'Meter Category' => $grandchild->meter_category,
                        'Installed On' => $grandchild->installed_on,
                    ]);
                }
            }
        }

        $headings = ['Global Device ID', 'Mode', 'Meter Type', 'Meter Category', 'Installed On'];

        return Excel::download(new DynamicTableExport($data, $headings), 'device_list.xlsx');
    }

}

